/* Copyright (c) 2017-2018 VMware, Inc. All rights reserved. -- VMware Confidential */
import {ProjectMigrator} from "./fileSystem/projectMigrator";
import * as yargs from "yargs";
import * as fs from "fs";
import * as path from "path";

const PROJECT_ARG: string = "project";
const MIGRATION_FILE_ARG: string = "migrationFile";
const MIGRATION_DESTINATION_ARG = "migrationDestination";
const OVERWRITE: string = "overwrite";

const options: { [key: string]: yargs.Options } = {
   project: {
      alias: "ms",
      describe: "The directory of the project which will be migrated.",
      demandOption: true
   },
   migrationFile: {
      alias: "mf",
      describe: "The destination where the migration file will be created.",
      demandOption: true
   },
   migrationDestination: {
      alias: "md",
      describe: "The destination directory where the project will be migrated.",
      demandOption: true
   },
   overwrite: {
      alias: "ow",
      describe: "overwrite existing files.",
      demandOption: false
   }
};

const commandDescription: string = "Migrate a project to a new one in which all calls to the bridge APIs are replaced with calls to the new JS APIs.\n" +
   "The migration can be done immediately or a migration patch file can be generated, edited and applied.\n" +
   "The original project is not edited. ";

yargs
   .usage(commandDescription)
   .command("migrateProject",
      "Copies the project to a new directory and migrates all Bridge API calls " +
      "to the new JS APIs in all .js, .ts and .html files. " +
      "The calls to the old Bridge APIs in the plugin.xml are also migrated.",
      (yargs: yargs.Argv) => {
         return yargs
            .option(PROJECT_ARG, options.project)
            .option(MIGRATION_DESTINATION_ARG, options.migrationDestination)
            .option(OVERWRITE, options.overwrite);
      },
      (argv: yargs.Arguments) => {
         migrateProject(argv.project, argv.migrationDestination, argv.overwrite);
      })
   .command("generateMigrationFile",
      "Generates a migration patch file that can be used to migrate the project. " +
      "The file can be edited before applying it.",
      (yargs: yargs.Argv) => {
         return yargs
            .option(PROJECT_ARG, options.project)
            .option(MIGRATION_FILE_ARG, options.migrationFile)
            .option(OVERWRITE, options.overwrite);
      },
      (argv: yargs.Arguments) => {
         generateMigrationFile(argv.project, argv.migrationFile, argv.overwrite);
      })
   .command("applyMigrationFile",
      "Migrates the project to a new directory with applied migration patch file.",
      (yargs: yargs.Argv) => {
         return yargs
            .option(PROJECT_ARG, options.project)
            .option(MIGRATION_DESTINATION_ARG, options.migrationDestination)
            .option(MIGRATION_FILE_ARG, options.migrationFile)
            .option(OVERWRITE, options.overwrite);
      },
      (argv: yargs.Arguments) => {
         applyMigrationFile(argv.project, argv.migrationFile, argv.migrationDestination, argv.overwrite);
      })
   .demandCommand().argv;

function generateMigrationFile(projectPath: string, migrationFile: string, overwrite: boolean): void {
   if (!directoryExists(projectPath)) {
      console.log(`The project '${projectPath} does not exist.`);
      return;
   }

   if(!directoryExists(path.parse(migrationFile).dir)) {
      console.log(`The directory '${path.parse(migrationFile).dir}' does not exists.`);
      return;
   }

   if (fileExists(migrationFile) && !overwrite) {
      console.log(`The file '${migrationFile}' already exists. Use --overwrite to overwrite.`);
      return;
   }

   if (fileExists(migrationFile)) {
      fs.unlinkSync(migrationFile);
   }

   const project = new ProjectMigrator(projectPath);
   project.createProjectTransformations(migrationFile);
}

function applyMigrationFile(projectPath: string, migrationFile: string, projectDestinationPath: string, overwrite: boolean): void {
   if (!directoryExists(projectPath)) {
      console.log(`The project '${projectPath} does not exist.`);
      return;
   }

   if (!fileExists(migrationFile)) {
      console.log(`The migration patch file '${migrationFile}' does not exist, generate it with the generateMigrationFile command.`);
      return;
   }

   if(!directoryExists(path.parse(projectDestinationPath).dir)) {
      console.log(`The directory '${path.parse(projectDestinationPath).dir}' does not exists.`);
      return;
   }

   if (overwrite) {
      deleteFolderRecursive(projectDestinationPath);
   }

   if (directoryExists(projectDestinationPath) && !isDirEmpty(projectDestinationPath)) {
      console.log(`The directory '${projectDestinationPath}' is not empty. Use --overwrite to overwrite.`);
      return;
   }

   ProjectMigrator.copyProject(projectPath, projectDestinationPath)
      .then(() => {
         const project = new ProjectMigrator(projectDestinationPath);
         try {
            project.applyProjectTransformations(migrationFile);
         } catch(e) {
            deleteFolderRecursive(projectDestinationPath);
            console.error(e);
         }
      });
}

function migrateProject(projectPath: string, projectDestinationPath: string, overwrite: boolean): void {
   if (!directoryExists(projectPath)) {
      console.log(`The project '${projectPath} does not exist.`);
      return;
   }

   if(!directoryExists(path.parse(projectDestinationPath).dir)) {
      console.log(`The directory '${path.parse(projectDestinationPath).dir}' does not exists.`);
      return;
   }

   if (overwrite) {
      deleteFolderRecursive(projectDestinationPath);
   }

   if (directoryExists(projectDestinationPath) && !isDirEmpty(projectDestinationPath)) {
      console.log(`The directory '${projectDestinationPath}' is not empty. Use --overwrite to overwrite.`);
      return;
   }

   ProjectMigrator.copyProject(projectPath, projectDestinationPath)
      .then(() => {
         const project = new ProjectMigrator(projectDestinationPath);
         try {
            project.migrateProject();
         } catch(e) {
            deleteFolderRecursive(projectDestinationPath);
            console.error(e);
         }
      });
}

function directoryExists(path: string): boolean {
   if (!(fs.existsSync(path) && fs.lstatSync(path).isDirectory())) {
      return false;
   }
   return true;
}

function fileExists(path: string): boolean {
   return ((fs.existsSync(path) && !fs.lstatSync(path).isDirectory()));
}

function isDirEmpty(path: string): boolean {
   return fs.readdirSync(path).length == 0;
}

function deleteFolderRecursive(path: string) {
   if (fs.existsSync(path)) {
      fs.readdirSync(path).forEach(function (file, index) {
         const curPath = path + "/" + file;
         if (fs.lstatSync(curPath).isDirectory()) { // recurse
            deleteFolderRecursive(curPath);
         } else { // delete file
            fs.unlinkSync(curPath);
         }
      });
      fs.rmdirSync(path);
   }
}